/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
*  Beck Computer-Loesungen GmbH
*  Garbenheimer Str. 30-38
*  D-35578 Wetzlar
*
*  Phone : (49)-6441-905-240
*  Fax   : (49)-6441-905-245
*
* ---------------------------------------------------------------------------
*
* ATTENTION : This library will only work with BC01-Firmware V1.10 and above
*
* ---------------------------------------------------------------------------
* Module        :    BC01.C
* Function      :    BC01/BC02 interface standard functions
* Author        :    tw/dg
* Date          :    10.06.97
* Version       :    1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author	Changes
*  0.01   11.02.97  vb     	first version
*  0.02   20.02.97  vb     	run-stop-switch,version,platform
*  0.03   21.02.97  vb     	watchdog-mode, error A/B-direction
*                          	all-disable(adjust,fast_counter,A/B-detection)
*  0.04   26.02.97  vb     	interrupt/counter input functions
*							(enable,disable,getvect,setvect)
*							a/b incremental input
*							a/b-init,a/b-deinit,interrupt-deinit,
*							fast_counter-deinit, a/b-reset,
*							fast_counter-reset
*  0.05   28.02.97	vb      fast_count-on,fast_count-off
*  0.06   18.04.97  dg      renamed to BC0X.C
*							remote functions added
*							geninterrupt() replaced bei int86/int86x
*  0.07   07.05.97  dg      timer services added
*  0.08   21.05.97  dg      static defines added
*  1.10   12.06.97  tw/dg   serial number added and version to 1.10
*                           include <string.h> added
*
****************************************************************************/


/****************************************************************************
* INCLUDES
****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <mem.h>
#include <string.h>

#ifdef __WATCOMC__
#include <i86.h>
//#define flags cflag
#endif

// BC0x-functions
#include "c:\automv6\c\BC01.H"

// define all local variables as static, so that they can be called
// from an timer or counter interrupt procedure
#define BC01_STATIC static
// define local variables not as static to save memory
//#define BC01_STATIC

/****************************************************************************
* Variables
****************************************************************************/

E_Type E;					// input image (structure)
A_Type A;					// output image (structure)
M_Type M;					// retentive flags (union)
BC01_Cfg_Type BC01_Cfg;		// BC01 configuration (structure)

long BC01_CycleTime;
BC01_RentOp_Type RentOp;	// structure for rententive operators/flags
Timer_Type Timer_Buffer[MAX_TIMER];

// communication
int BC01_Com_Result;

/****************************************************************************
*
* Start of BC01/BC0x low-level BIOS and service-interrupt functions
*
****************************************************************************/

/****************************************************************************
* Functions for Version_Information BC01
****************************************************************************/

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read hc0x/bc0x version information
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : VersBios : int *	(out hexadecimal)
*                 VersBC01 : int *	(out hexadecimal)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  vb
* Last Changes  : (date)
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  20.02.97   vb     first
****************************************************************************/

void BC01_GetVersion(int *VersBios,int *VersBC01)
{
	union REGS regs;

	regs.h.ah = 0xFD;
	int86(0x6F,&regs,&regs);
	*VersBios = regs.x.cx;
	*VersBC01 = regs.x.ax;
} // BC01_GetVersion()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Hardware platform detection
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : Platf : char *
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 20.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  20.02.97   vb     first version
****************************************************************************/

void BC01_GetPlatform(char *Platf)
{
	union REGS regs;

	regs.x.bx = 0xFACE;
	regs.x.ax = 0xFF55;
	int86(0x6F,&regs,&regs);
	Platf[0] = regs.h.ch;
	Platf[1] = regs.h.cl;
	Platf[2] = regs.h.ah;
	Platf[3] = regs.h.al;
    Platf[4] = '\0';

} // BC01_GetPlatform()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Get Serial number of BC01
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : Serial : char * : if not avail or not programmed Serial is
*									"000000"
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : tw
* Last Changes  : 10.06.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  10.06.97   vb     first version
****************************************************************************/

void BC01_GetSerial(char *Serial)
{
	union REGS regs;
    regs.x.ax = 0xfb00;
	int86(0x6F,&regs,&regs);
	if (regs.x.cflag)
    {
    	strcpy(Serial,"000000");
    }
    if (regs.x.ax | regs.x.cx)
    {
		Serial[0] = (regs.h.cl >> 4) + 0x30;
		Serial[1] = (regs.h.cl & 0x0f) + 0x30;
		Serial[2] = (regs.h.ah >> 4) + 0x30;
		Serial[3] = (regs.h.ah & 0x0f) + 0x30;
		Serial[4] = (regs.h.al >> 4) + 0x30;
		Serial[5] = (regs.h.al & 0x0f) + 0x30;
        Serial[6] = '\0';
    }
    else
    {
    	strcpy(Serial,"000000");
    }
} // BC01_GetSerial

/****************************************************************************
* Functions for BC0X-WATCHDOG
****************************************************************************/

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Set watchdog to user-mode
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : (date)
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00   11.02.97  vb     first
****************************************************************************/

void BC01_Watchdog_SetUser(void)
{
	BC01_STATIC union REGS regs;

	regs.h.ah = 0xA4;		// function 0xA4 - watchdog
	regs.h.al = 0x02;		// set watchdog mode 2 (user)
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : user watchdog refresh
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : (date)
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00   11.02.97  vb     first
****************************************************************************/
void BC01_Watchdog_Refresh(void)
{
	BC01_STATIC union REGS regs;

	regs.h.ah = 0xA5;			// function 0xA5
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Set watchdog to BIOS-Mode
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : (date)
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00   11.02.97  vb     first
****************************************************************************/

void BC01_Watchdog_SetBios(void)
{
	BC01_STATIC union REGS regs;

	regs.h.ah = 0xA4;			// function 0xA3
	regs.h.al = 0x03;			// automatic refresh by BIOS
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read watchdog mode
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : watchdog mode:int   (2|3)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : (date)
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00   21.02.97  vb     first
****************************************************************************/

int BC01_Watchdog_GetMode(void)
{
	union REGS regs;

	regs.x.ax = 0xA4;				// function 0xA4
	regs.h.al = 0x00;				// get mode
	int86(0x6F,&regs,&regs);

	return regs.h.al;
}

/****************************************************************************
* Functions for DUO-LED
****************************************************************************/

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : set LED
* Prototype     : (as listed in header-file)
* Public/Local  : public  (public means prototype is in header-file)
* Parameters    : number: int  (0..7)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 21.02.97
* Version       : 1.01
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*   1.00  11.02.97  vb     first
*   1.01  21.02.97  vb     set_LED
****************************************************************************/

void BC01_Led_Write(int Value)
{
	BC01_STATIC union REGS regs;
	BC01_STATIC struct SREGS sregs;
	BC01_STATIC int Buffer;

	Buffer = Value;					// value != 0xFF: write LED
	regs.x.ax = 0xD100;				// function 0xD100
	regs.x.cx = 1;					// write 1 byte
	regs.x.dx = 0x0100;				// sub-function 0x0100
	sregs.es  = FP_SEG(&Buffer);	// pointer to buffer
	regs.x.bx = FP_OFF(&Buffer);
	int86x(0x6F,&regs,&regs,&sregs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read LED status
* Prototype     : (as listed in header-file)
* Public/Local  : public (public means prototype is in header-file)
* Parameters    : -/-
* Return        : number: char
* Header-File   :  BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 21.02.97
* Version       : 1.01
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*   1.00  11.02.97  vb     first
*   1.01  21.02.97  vb     read_LED_status
****************************************************************************/

int BC01_Led_Read(void)
{
	BC01_STATIC union REGS regs;
	BC01_STATIC struct SREGS sregs;
	BC01_STATIC int Buffer;

	Buffer = 0xFF;					// value==0xFF: read LED
	sregs.es  = FP_SEG(&Buffer);	// pointer to buffer
	regs.x.bx = FP_OFF(&Buffer);
	regs.x.ax = 0xD100;				// function 0xD100
	regs.x.cx = 1;					// read 1 byte
	regs.x.dx = 0x0100;				// sub-function 0x0100
	int86x(0x6F,&regs,&regs,&sregs);
	Buffer = regs.h.al;

	return Buffer;
}

/****************************************************************************
* Functions for RUN-STOP-SWITCH
****************************************************************************/

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read run/stop switch
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : switch-position:int      (0|1)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  vb
* Last Changes  :  20.02.97
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  20.02.97   vb     first
****************************************************************************/

int BC01_RunStop_Read(void)
{
	BC01_STATIC union REGS regs;
	BC01_STATIC struct SREGS sregs;
	BC01_STATIC int buffer;

	buffer = 0;
	sregs.es  = FP_SEG(&buffer);
	regs.x.bx = FP_OFF(&buffer);
	regs.h.ah = 0xD0;			// function 0xD000
	regs.h.al = 0x00;			// byte-mode
	regs.x.cx = 1;				// read 2 byte
	regs.x.dx = 0x0100;			// sub-function 0x01, read run/stop
	int86x(0x6F,&regs,&regs,&sregs);

	return buffer;
}

/****************************************************************************
* Functions for BC01-INPUTS/OUTPUTS
****************************************************************************/

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read digital input
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : buffer	: BYTE *
*                 size		: int (= length destination-buffer)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.97
* Version       : 1.01
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb     first
****************************************************************************/

void BC01_Inputs_Read(void *buffer, int size)
{
	BC01_STATIC union REGS regs;
	BC01_STATIC struct SREGS sregs;

	sregs.es  = FP_SEG(buffer);
	regs.x.bx = FP_OFF(buffer);
	regs.h.ah = 0xD0;				// function 0xD0
	regs.h.al = 0x00;		  		// 0x00 = byte-mode
	regs.x.cx = size;     			// number of bytes to read
	regs.x.dx = 0x0000;				// 0x0000 = local digital inputs
	int86x(0x6F,&regs,&regs,&sregs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Write to out
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : buffer	: BYTE *,
*				  size		: int (= length source-buffer)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  vb
* Last Changes  : (date)
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb     first
****************************************************************************/

void BC01_Outputs_Write(void *buffer, int size)
{
	BC01_STATIC union REGS regs;
	BC01_STATIC struct SREGS sregs;

	sregs.es  = FP_SEG(buffer);
	regs.x.bx = FP_OFF(buffer);
	regs.h.ah = 0xD1;			// functiob 0xD1
	regs.h.al = 0x00;         	// 0x00 = byte mode
	regs.x.cx = size;         	// number of bytes to write
	regs.x.dx = 0x0000;			// write local digital outputs
	int86x(0x6F,&regs,&regs,&sregs);
}

/****************************************************************************
* Functions for FAST-COUNTERS
****************************************************************************/

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : get number of fast counters available
* Prototype     : (as listed in header-file)
* Public/Local  : public (public means prototype is in header-file)
* Parameters    : -/-
* Return        : number : int     (2)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*   1.00  11.02.97  vb      first version
****************************************************************************/

int BC01_FastCounter_Info(void)
{
	union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x00;			// sub-function 0x00
	int86(0x6F,&regs,&regs);

	return regs.h.ah;			// return number of fast counters
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : enable fast counters
* Prototype     : (as listed in header-file)
* Public/Local  : public  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 28.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  28.02.97  vb      first version
****************************************************************************/

void BC01_FastCounter_Enable(void)
{
	union REGS regs;

	regs.h.ah = 0xD0;			// function 0xD0
	regs.h.al = 0x00;
	regs.x.dx = 0x0380;			// sub-function 0x0380
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : disable fast counters
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : -/-
* Header-File   :  BC01.H
* ---------------------------------------------------------------------------
* Author        : vb/dg
* Last Changes  : 28.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  28.02.97  vb      first version
****************************************************************************/

void BC01_FastCounter_Disable(void)
{
	union REGS regs;

	regs.h.ah = 0xD0;			// function 0xD0
	regs.h.al = 0x00;
	regs.x.dx = 0x0381;			// sub-function 0x0381
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : init fast counter
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : counter: int            (counter number),
*                 mode: int                (0-single|1-continious),
*                 max : unsigned int       (max. counter value)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb      first
****************************************************************************/

void BC01_FastCounter_Init(int n, int Mode, WORD Max)
{
	union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x05;			// sub-function 0x05 (init/deinit)
	regs.h.dh = 0x01;			// ???
	regs.h.ch = 0x01;			// 0x01=init, 0x00=deinit
	regs.h.dl = n;				// 0=E1.2 / 1=E1.3
	regs.x.bx = Max;			// max. counter value
	regs.h.cl = Mode;			// single/continious
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : deinit fast counter
* Prototype     : (as listed in header-file)
* Public/Local  : public  (public means prototype is in header-file)
* Parameters    : counter number: int     (0|1),
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : (date)
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  26.02.97   vb      first
****************************************************************************/

void BC01_FastCounter_Deinit(int n)	// ???
{
	union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x05;			// sub-function 0x05 (init/deinit)
	regs.h.ch = 0x00;			// 0x00 = deinit, 0x01=init
	regs.h.dh = 0x01;			// ???
	regs.h.dl = n;				// 0=E1.2 / 1=E1.3
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : start fast counter
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : counter number:int (0|1)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.07
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97  vb      first version
****************************************************************************/

void BC01_FastCounter_Start(int n)
{
	BC01_STATIC union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x06;			// sub-function 0x06
	regs.h.dh = 0x01;			// ???
	regs.h.dl = n;				// 0=E1.2 / 1=E1.3
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : stop fast counter
* Prototype     : (as listed in header-file)
* Public/Local  : public (public means prototype is in header-file)
* Parameters    : counter number: int (0|1)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb      first version
****************************************************************************/

void BC01_FastCounter_Stop(int n)
{
	BC01_STATIC union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x07;			// sub-function 0x07
	regs.h.dh = 0x01;			// ???
	regs.h.dl = n;				// 0=E1.2 / 1=E1.3
	int86(0x6F,&regs,&regs);
}
/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read fast counter value
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : counter number:int    (0|1)
* Return        : counter position: unsigned int
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb      first
****************************************************************************/

WORD BC01_FastCounter_Read(int n)
{
	BC01_STATIC union REGS regs;
	BC01_STATIC struct SREGS sregs;
	BC01_STATIC WORD Buffer = 0;

	sregs.es  = FP_SEG(&Buffer);		// pointer to value
	regs.x.bx = FP_OFF(&Buffer);
	regs.h.ah = 0xD0;				// function 0xD0
	regs.h.al = 0x00;				// byte mode
	regs.x.cx = 2;					// read 2 bytes
	regs.h.dh = 0x03;				// sub-function 0x03, read counter
	regs.h.dl = n;					// 0=E1.2 / 1=E1.3
	int86x(0x6F,&regs,&regs,&sregs);

	return Buffer;
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : reset fast counter to zero
* Prototype     : (as listed in header-file)
* Public/Local  : public  (public means prototype is in header-file)
* Parameters    : counter number:int (0|1)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 26.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  26.02.97  vb      first
****************************************************************************/

void BC01_FastCounter_SetZero(int n)
{
	BC01_STATIC union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x09;			// sub-function 0x09
	regs.h.dh = 0x01;			// ???
	regs.h.dl = n;				// 0=E1.2 / 1=E1.3
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
* Functions for TRIMMER
****************************************************************************/

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Potentiometer input on
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb     first
****************************************************************************/

void BC01_Trimmer_Enable(void)
{
	union REGS regs;

	regs.h.ah = 0xD4;				// function 0xD4
	regs.h.al = 0x81;				// sub-function 0x81
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : potentiometer input off
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : (date)
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb     first
****************************************************************************/
void BC01_Trimmer_Disable(void)
{
	union REGS regs;

	regs.h.ah = 0xD4;			// function 0xD4
	regs.h.al = 0x80;			// sub-function 0x80
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read potentiometer value
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : adjust position: int
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : (date)
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb     first
****************************************************************************/
int BC01_Trimmer_Read(void)
{
	BC01_STATIC union REGS regs;
	BC01_STATIC struct SREGS sregs;
	BC01_STATIC int Buffer;

	Buffer = 0;
	sregs.es  = FP_SEG(&Buffer);
	regs.x.bx = FP_OFF(&Buffer);
	regs.h.ah = 0xD0;				// function 0xD0
	regs.h.al = 0x00;				// byte-mode
	regs.x.cx = 2;				// read 2 bytes
	regs.x.dx = 0x0200;			// sub-function read poti 0
	int86x(0x6F,&regs,&regs,&sregs);

	return Buffer;
}

/****************************************************************************
* Functions for A/B-INPUTS
****************************************************************************/

/****************************************************************************
* Functions for A/B-INPUTS
****************************************************************************/

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : initalize A/B incremental counter
* Prototype     : (as listed in header-file)
* Public/Local  : public  (public means prototype is in header-file)
* Parameters    : RefValue: int
* Return        :    -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : (date)
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00   26.02.97  vb      first
****************************************************************************/

void BC01_ABcount_Init(int RefValue)
{
	union REGS regs;

	regs.x.ax = 0xD505;
	regs.x.bx = RefValue;
	regs.x.cx = 0x101;
	regs.x.dx = 0;
	int86(0x6F,&regs,&regs);
	regs.x.ax = 0xD505;
	regs.x.bx = RefValue;
	regs.x.cx = 0x101;
	regs.x.dx = 0x01;
	int86(0x6F,&regs,&regs);
	BC01_Interrupt_Enable(0);
	BC01_Interrupt_Enable(1);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : deinitalise A/B incremental counter
* Prototype     : (as listed in header-file)
* Public/Local  : public  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 26.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00   26.02.97  vb      first
****************************************************************************/

void BC01_ABcount_Deinit(void)
{
	union REGS regs;

	regs.x.ax = 0xD000;
	regs.x.dx = 0x480;
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : set A/B incremental counter to RefValue
* Prototype     : (as listed in header-file)
* Public/Local  : public  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 26.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00   26.02.97  vb      first version
****************************************************************************/

void BC01_ABcount_Reset(void)
{
	BC01_STATIC union REGS regs;

	regs.x.ax = 0xD509;
	regs.h.dh = 0;
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read A/B incremental counter value
* Prototype     : (as listed in header-file)
* Public/Local  : public  (public means prototype is in header-file)
* Parameters    : error: int *     (0 no error, 1 overflow , -1 underflow)
* Return        : counter-position: int
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb      first
****************************************************************************/

int BC01_ABcount_Read(int *error)
{
	BC01_STATIC union REGS regs;
	BC01_STATIC struct SREGS sregs;
	BC01_STATIC int count;

	sregs.es  = FP_SEG(&count);
	regs.x.bx = FP_OFF(&count);
	regs.h.ah = 0xD0;				// function 0xD0
	regs.h.al = 0x00;				// byte-mode
	regs.x.dx = 0x0400;			// sub-function 0x0400, read a/b
	regs.x.cx = 2;				// read 2 bytes
	int86x(0x6F,&regs,&regs,&sregs);
	*error = regs.x.ax;

	return count;
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read A/B incremental counter direction
* Prototype     : (as listed in header-file)
* Public/Local  : public (public means prototype is in header-file)
* Parameters    : -/-
* Return        : direction-position:int    (0|1 example 0-A 1-B)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97  vb      first version
****************************************************************************/

int BC01_ABcount_ReadDirection(void)
{
	BC01_STATIC union REGS regs;
	BC01_STATIC struct SREGS sregs;
	BC01_STATIC int dir;

	dir = 0;
	sregs.es = FP_SEG(&dir);
	regs.x.bx = FP_OFF(&dir);
	regs.h.ah = 0xD0;				// function 0xD0
	regs.h.al = 0x00;				// byte-mode
	regs.x.dx = 0x0401;			// sub-function 0x0401
	regs.x.cx = 2;				// read 2 bytes
	int86x(0x6F,&regs,&regs,&sregs);

	return dir;
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : set A/B reference
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : number :int   (0-7),
				  bit0: 0=no reference setting / 1=reference setting
				  bit1: 0=set ref. only one time / 1=allways set ref.
				  bit2: 0=set ref. on direction=0 / 1=set ref. on direction=1
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  vb
* Last Changes  : (date)
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  09.06.97   dg     first
****************************************************************************/

void BC01_ABcount_SetReferenceMode(int ReferenceMode)
{
	union REGS regs;

	regs.h.ah = 0xD5;				// function 0xD5
	regs.h.al = 0x0B;				// sub-function 0x0B
	regs.x.cx = ReferenceMode & 0x07;
        regs.x.dx = 0;
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
* Functions for REMOTE-IO
****************************************************************************/

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : check if remote driver REO.EXE is installed
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : int	0 = host
						1 = remote
						-1 = stand-alone (not initialised)
						-2 = driver not installed
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  dg
* Last Changes  :  21.04.97
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  18.04.97   dg     first version
****************************************************************************/

int BC01_Remote_Installed(void)
{
	union REGS regs;

	regs.h.ah = 0xD0;				// function 0xD0
	regs.h.dh = 0x80;				// sub-function 0x80, remote
	regs.h.dl = 0x00;				//
	int86(0x6F,&regs,&regs);
	if (regs.x.ax==0xA55A)
		// driver is installed, return state
		return regs.x.cx;
	else
		// driver is not installed
		return -2;
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : initialise remote driver and set to master mode
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : remote_count: int
* Return        : int (1=no error / 0=error - slaves not found)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  dg
* Last Changes  :  18.04.97
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  18.04.97   dg     first version
****************************************************************************/

int BC01_Remote_Init(int remote_count)
{
	union REGS regs;

	regs.h.ah = 0xD0;				// function 0x00
	regs.h.dh = 0x80;				// sub-function 0x80
	regs.h.dl = remote_count;		//
	int86(0x6F,&regs,&regs);

	return !regs.x.cflag;			// cflag==1: error
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read run/stop switch, local and remote
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : remote:int (0=local, 0xFF0n=remote, n=0..2)
* Return        : switch-position:int      (0|1)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  dg
* Last Changes  :  18.04.97
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  18.04.97   dg     first
****************************************************************************/

int BC01_RunStop_ReadRemote(int remote)
{
	BC01_STATIC union REGS regs;
	BC01_STATIC struct SREGS sregs;
	BC01_STATIC int buffer;

	buffer = 0;
	sregs.es  = FP_SEG(&buffer);
	regs.x.bx = FP_OFF(&buffer);
	regs.h.ah = 0xD0;				// function 0xD0
	regs.h.al = 0x00;				// byte-mode
	regs.x.cx = 1;					// read 1 byte
	if (!remote)
		regs.x.dx = 0x0100;			// sub-function: local
	else
	{
		regs.h.dh = 0xFE;			// sub-function: remote
		regs.h.dl = remote-1;		// number of remote station
	}
	int86x(0x6F,&regs,&regs,&sregs);

	return buffer;

}	// BC01_RunStop_ReadRemote()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read local and remote inputs
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : read-category:int(0-Byte,1-Word Hi-Byte=0, 2-Word Hi-Byte=1),
*					   destination-buffer:unsigned char pointer
*                 read-bytes-number:int (= length destination-buffer)
*						remote:int (0=local, 0xFF0n=remote, n=0..2)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 18.04.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  19.04.97   dg     first version
****************************************************************************/

void BC01_Inputs_ReadRemote(void *buffer, int size, int remote)
{
	BC01_STATIC union REGS regs;
	BC01_STATIC struct SREGS sregs;

	sregs.es = FP_SEG(buffer);
	regs.x.bx = FP_OFF(buffer);
	regs.h.ah = 0xD0;			// function 0xD0xx
	regs.h.al = 0x00;		  	// byte-mode
	regs.x.cx = size;     	// number of bytes to read
	if (!remote)
		regs.x.dx = 0x0000;	// local
	else
	{
		regs.h.dh = 0xFF;		// remote
		regs.h.dl = remote-1;	// remote station number (0..n-1)
	}
	int86x(0x6F,&regs,&regs,&sregs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : write to digital output (local and remote)
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : write-category: int ( 0-Byte,1-Word ),
*				  source-buffer: unsigned char pointer ,
*				  write-bytes-number:int (= length source-buffer)
*				  remote:int (0=local, 1..3=remote)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  dg
* Last Changes  :  18.04.97
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  18.04.97   dg     first
****************************************************************************/

void BC01_Outputs_WriteRemote(void *buffer, int size, int remote)
{
	BC01_STATIC union REGS regs;
	BC01_STATIC struct SREGS sregs;

	sregs.es = FP_SEG(buffer);
	regs.x.bx = FP_OFF(buffer);
	regs.h.ah = 0xD1;
	regs.h.al = 0x00;         // byte mode
	regs.x.cx = size;         // number of bytes to write
	if (!remote)
		regs.x.dx = 0x0000;	// local
	else
	{
		regs.h.dh = 0xFF;		// remote
		regs.h.dl = remote-1;
	}
	int86x(0x6F,&regs,&regs,&sregs);
}

/****************************************************************************
* Functions for INTERRUPT-INPUTS
****************************************************************************/

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read number of interrupt inputs available
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : number:int      (2)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  vb
* Last Changes  : (date)
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb     first
****************************************************************************/

int BC01_IntrInputs_Info(void)
{
	union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x00;			// sub-function 0x00
	int86(0x6F,&regs,&regs);

	return regs.h.al;
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : configure input E1.n as interrupt input
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : input number: int    (0|1)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb     first version
****************************************************************************/

void BC01_IntrInputs_Enable(int n)
{
	union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x05;			// sub-function 0x05 (init/deinit)
	regs.h.dh = 0x00;			// interrupt input
	regs.x.cx = 0x0100;			// 0x0100=init / 0x0000=deinit
	regs.h.dl = n;				// 0=E1.0 / 1=E1.1
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : configure E1.n as standard digital input
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : input number: int    (0|1)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 26.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  26.02.97   vb     first
****************************************************************************/

void BC01_IntrInputs_Disable(int n)
{
	union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x05;			// sub-function 0x05 (init/deinit)
	regs.h.dh = 0x00;			// interrupt input
	regs.x.cx = 0x0000;			// 0x0000=deinit, 0x0100=init
	regs.h.dl = n;				// 0=E1.0 / 1=E1.1
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read interrupt vector for interrupt input
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : input number: int         (0|1)
* Return        : pointer :void interrupt  (oldhandler)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb     first
****************************************************************************/

void interrupt (*BC01_IntrInputs_GetVect(int n))(void)
{
	union REGS regs;
	void interrupt far (*oldhandler)(void);

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x01;			// sub-function 0x01 (getvect)
	regs.h.dh = 0x00;			// interrupt input
	regs.h.dl = n;			// 0=E1.0 / 1=E1.1
	int86(0x6F,&regs,&regs);
	oldhandler = MK_FP(regs.x.cx,regs.x.ax);

	return oldhandler;
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : set interrupt service routine for interrupt input
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : input number:int   (0|1),
*                 pointer :void interrupt      (handler)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  28.04.97   dg     first
****************************************************************************/

void BC01_IntrInputs_SetVect(int n, void interrupt (*handler)())
{
	union REGS regs;
	struct SREGS sregs;

	sregs.es  = FP_SEG(handler);	// pointer to interrupt handler
	regs.x.bx = FP_OFF(handler);
	regs.h.ah = 0xD5;				// function 0xD5
	regs.h.al = 0x02;				// sub-function 0x02 (setvect)
	regs.h.dh = 0x00;				// interrupt input
	regs.h.dl = n;					// 0=E1.0 / 1=E1.1
	int86x(0x6F,&regs,&regs,&sregs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : enable interrupt for interrupt input E1.n
* Prototype     : (as listed in header-file)
* Public/Local  : public (public means prototype is in header-file)
* Parameters    : input number:int (0|1)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 11.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb     first
****************************************************************************/

void BC01_Interrupt_Enable(int n)
{
	union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x03;			// sub-function 0x03 (enable)
	regs.h.dh = 0x00;			// interrupt input
	regs.h.dl = n;				// 0=E1.0 / 1=E1.1
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : disable interrupt for interrupt input E1.n
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : input number: int (0|1)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  vb
* Last Changes  : (date)
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  11.02.97   vb     first
****************************************************************************/

void BC01_Interrupt_Disable(int n)
{
	union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x04;			// sub-function 0x04 (disable)
	regs.x.dx = 0x00;			// interrupt input
	regs.h.dl = n;			// 0=E1.0 / 1=E1.1
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : poti off, fast counter off, A/B incremental counter off
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 21.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  21.02.97   vb     first version
****************************************************************************/

void BC01_DisableAll(void)
{
	union REGS regs;

	regs.x.ax = 0xD000;
	regs.x.dx = 0x7FFF;
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : enable interrupt input
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : entry-number   (0|1)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : vb
* Last Changes  : 26.02.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  26.02.97   vb     first version
****************************************************************************/
void BC01_IntCount_Enable(int n)
{
	union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x03;			// sub-function 0x03
	regs.h.dh = 0x01;			// 0x01 = counter interrupt
	regs.h.dl = n;			// 0=E1.2 / 1=E1.3
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : disable interrupt input
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : entry-number   (0|1)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  vb
* Last Changes  : (date)
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  26.02.97   vb     first
****************************************************************************/

void BC01_IntCount_Disable(int n)
{
	union REGS regs;

	regs.h.ah = 0xD5;			// function 0xD4
	regs.h.al = 0x04;			// sub-function 0x04
	regs.h.dh = 0x01;			// 0x01 = counter interrupt
	regs.h.dl = n;			// 0=E1.2 / 1=E1.3
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : get pointer to service routine of interrupt input n
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : entry-number: int (0|1)
* Return        : pointer :void interrupt   (oldhandler)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  vb
* Last Changes  : (date)
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  26.02.97   vb     first
****************************************************************************/

void interrupt(*BC01_IntCount_GetVect(int n))(void)
{
	union REGS regs;
	void interrupt far (*oldhandler)(void);

	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x01;			// sub-function 0x01
	regs.h.dh = 0x01;			// 0x01 = counter interrupt
	regs.h.dl = n;			// 0=E1.2 / 1=E1.3
	int86(0x6F,&regs,&regs);
	oldhandler = MK_FP(regs.x.cx,regs.x.ax);

	return oldhandler;
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : set service routine for interrupt input n
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : entry-number :int   (0|1),
*                 pointer : void interrupt      (handler)
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  vb
* Last Changes  : (date)
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  26.02.97   vb     first
****************************************************************************/

void BC01_IntCount_SetVect(int n,void interrupt (*handler)())
{
	union REGS regs;
	struct SREGS sregs;

	sregs.es  = FP_SEG(handler);
	regs.x.bx = FP_OFF(handler);
	regs.h.ah = 0xD5;			// function 0xD5
	regs.h.al = 0x02;			// sub-function 0x01
	regs.h.dh = 0x01;			// 0x01 = counter interrupt
	regs.h.dl = n;			// 0=E1.2 / 1=E1.3
	int86x(0x6F,&regs,&regs,&sregs);
}

/****************************************************************************
*  Timer/Counter-Services
****************************************************************************/

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : check if timer/counter service is available
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : int: 0=not available, 1=available
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 30.04.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  30.04.97   dg     first version
****************************************************************************/

BOOL BC01_Timer_Installed(void)
{
	union REGS regs;

	regs.h.ah = 0xC0;			// function 0xC0
	regs.h.al = 0x55;			// should return a 0xAA in the AL register
	int86(0x6F,&regs,&regs);
	//printf("Timer_Installed: CF=%02X AL=%02X\n",regs.x.cflag,regs.h.al);
	if (regs.x.cflag==0 && regs.h.al==0xAA)
		return 1;	// available
	else
		return 0;	// not available
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : enable timer service
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : TimerType *Buffer: pointer to timer buffer
*               : int n: number of timer
* Return        : int: 0=no error, 1=error
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 07.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  07.05.97   dg     first version
****************************************************************************/

BOOL BC01_Timer_Enable(Timer_Type *Buffer, int n)
{
	union REGS regs;
	struct SREGS sregs;

	if (n==0)
		return 1;

	regs.h.ah = 0xC1;				// function 0xC1
	regs.h.al = n;					// number of timer
	sregs.es  = FP_SEG(Buffer);
	regs.x.bx = FP_OFF(Buffer);
	int86x(0x6F,&regs,&regs,&sregs);
	//printf("Timer_Enable: CF=%02X\n",regs.x.cflag);
	return !regs.x.cflag;
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : disable timer service
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : TimerType *Buffer: pointer to timer buffer
*               : int n: number of timer
* Return        : int: 0=no error, 1=error
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 07.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  07.05.97   dg     first version
****************************************************************************/

void BC01_Timer_Disable(void)
{
	union REGS regs;

	regs.h.ah = 0xC2;				// function 0xC2
	int86(0x6F,&regs,&regs);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : insert a new 32 bit timer
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int ContMode
*               : int IncMode
*               : WORD Prescale
* Return        : int: handle (timer number)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 07.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  07.05.97   dg     first version
****************************************************************************/

int BC01_Timer_Insert(int ContMode, int IncMode, WORD Prescale)
{
	union REGS regs;

	regs.h.ah = 0xC4;			// function 0xC4
	regs.h.al = ContMode;		// counter mode
								// 0: continous, >0: number of periods
	regs.h.bl = IncMode;		// 0: increment
								// 1: increment until 0
								// 2: decrement
								// 3: decrement until 0
	regs.x.cx = Prescale;		// number of 4.8 ms ticks per count
	int86(0x6F,&regs,&regs);
	//printf("Timer_Insert: CF=%02X AL=%02X\n",regs.x.cflag,regs.h.al);
	if (regs.x.cflag==0)
		return regs.h.al+1;		// ok: return handle
	else
		return 0;				// error
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : set timer increment mode
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int Handle: timer number
*               : int IncMode: count mode
* Return        : BOOL: 0 = error, 1 = ok
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 07.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  07.05.97   dg     first version
****************************************************************************/

int BC01_Timer_SetIncr(int Handle, int IncMode)
{
	union REGS regs;

	regs.h.ah = 0xC5;			// function 0xC4
	regs.h.al = Handle-1;		// timer number
	regs.h.bl = IncMode;		// 0: increment
								// 1: increment until 0
								// 2: decrement
								// 3: decrement until 0
	int86(0x6F,&regs,&regs);
	return !regs.x.cflag;
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : get timer periode
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int Handle: timer number
* Return        : BOOL: 0 = error, >0 = prescale value
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 07.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  07.05.97   dg     first version
****************************************************************************/

WORD BC01_Timer_GetPrescale(int Handle)
{
	union REGS regs;

	regs.h.ah = 0xC6;			// function 0xC4
	regs.h.al = Handle-1;		// timer number
	regs.h.bl = 0;				// sub-function 0: read prescale value
	int86(0x6F,&regs,&regs);
	if (!regs.x.cflag)
		return regs.x.cx;		// return prescale value
	else
		return 0;				// error
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : set timer periode
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int Handle: timer number
*               : WORD Prescale: prescale value
* Return        : BOOL: 0 = error, 1 = ok
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 07.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  07.05.97   dg     first version
****************************************************************************/

BOOL BC01_Timer_SetPrescale(int Handle, WORD Prescale)
{
	union REGS regs;

	regs.h.ah = 0xC6;			// function 0xC4
	regs.h.al = Handle-1;		// timer number
	regs.h.bl = 2;				// sub-function 2: set prescale value
	regs.x.cx = Prescale;
	int86(0x6F,&regs,&regs);
	return !regs.x.cflag;		// 0 = error, 1= ok
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : start timer
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int Handle: timer number
* Return        : BOOL: 0 = error, 1 = ok
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 07.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  07.05.97   dg     first version
****************************************************************************/

BOOL BC01_Timer_Start(int Handle)
{
	BC01_STATIC union REGS regs;

	regs.h.ah = 0xC7;			// function 0xC7
	regs.h.al = Handle-1;		// timer number
	int86(0x6F,&regs,&regs);
	//printf("Timer_Start: CF=%02X\n",regs.x.cflag);
	return !regs.x.cflag;		// 0 = error, 1= ok
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : stop timer
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int Handle: timer number
* Return        : BOOL: 0 = error, 1 = ok
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 07.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  07.05.97   dg     first version
****************************************************************************/

BOOL BC01_Timer_Stop(int Handle)
{
	BC01_STATIC union REGS regs;

	regs.h.ah = 0xC8;			// function 0xC8
	regs.h.al = Handle-1;		// timer number
	int86(0x6F,&regs,&regs);
	//printf("Timer_Stop: CF=%02X\n",regs.x.cflag);
	return !regs.x.cflag;		// 0 = error, 1= ok
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read timer value
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int Handle: timer number
* Return        : BOOL: timer value, 0xFFFFFFFF = error
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 07.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  07.05.97   dg     first version
****************************************************************************/

DWORD BC01_Timer_Read(int Handle)
{
	BC01_STATIC union REGS regs;

	regs.h.ah = 0xCB;			// function 0xCB
	regs.h.al = Handle - 1;		// timer number
	int86(0x6F,&regs,&regs);
	if (!regs.x.cflag)
								// timer value
		return (((DWORD)regs.x.cx)<<16) | ((DWORD)regs.x.ax);
	else
		return 0xFFFFFFFFl;		// error
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : read timer value
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int Handle: timer number
* Return        : BOOL: timer value, 0xFFFF = error
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 07.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  07.05.97   dg     first version
****************************************************************************/

BOOL BC01_Timer_Write(int Handle, DWORD Value)
{
	BC01_STATIC union REGS regs;

	regs.h.ah = 0xCA;					// function 0xCA
	regs.h.al = Handle-1;				// timer number
	regs.x.cx = (WORD) (Value>>16);		// high word
	regs.x.bx = (WORD) (Value&=0x0000FFFFl);	// low word
	int86(0x6F,&regs,&regs);
	return !regs.x.cflag;				// 0 = error, 1= ok
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : set timer interrupt
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int CntrMode:
*               : WORD Prescale:
*               : void interrupt (*handler)()): pointer to interrupt routine
* Return        : int:
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 07.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  07.05.97   dg     first version
****************************************************************************/

int BC01_TimerIntr_SetVect(int CntrMode,WORD Prescale,
							void interrupt (*handler)())
{
	union REGS regs;
	struct SREGS sregs;

	regs.h.ah = 0xC3;			// function 0xCB
	regs.h.al = CntrMode;		// 0 = continous, >0 = number of events
	regs.x.cx = Prescale;
	sregs.es  = FP_SEG(handler);	// pointer to interrupt handler
	regs.x.bx = FP_OFF(handler);
	int86x(0x6F,&regs,&regs,&sregs);
	//printf("TimerIntr_SetVect: CF=%02X AL=%02X\n",regs.x.cflag,regs.h.al);
	if (!regs.x.cflag)
		return regs.h.al+1;		// handle
	else
		return 0;				// error
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : get BIOS timer value, 55ms = 18.2 hertz
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -
* Return        : long:
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

long BC01_BiosTimer_Get(void)
{
	long far *timer;

	timer = MK_FP(0x40,0x6C);
	return *timer;
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : wait until BIOS timer reaches the value biostime+wait
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : long: BiosTime
*				  int: Wait
* Return        : int:
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

int BC01_BiosTimer_Wait(long BiosTime, int Wait)
{
	long t;

	t = BC01_BiosTimer_Get();

	if (BiosTime>t)
		// bios timer reset over midnight
		return 0;

	// wait
	while (BC01_BiosTimer_Get()<BiosTime+Wait);

	return (int)(BC01_BiosTimer_Get()-t);
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : wait for n BIOS-timer ticks
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: Ticks
* Return        : -/-
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 27.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  27.05.97   dg     first version
****************************************************************************/

void BC01_BiosTimer_Delay(int Ticks)
{
	long t;

	t = BC01_BiosTimer_Get();

	// wait
	while (BC01_BiosTimer_Get()<t+Ticks)
	{
		if (BC01_BiosTimer_Get()<t)
			// bios timer reset over midnight
			return;
	}

}

/****************************************************************************
*
* Start of high-level hc0x/bc0x functions
*
****************************************************************************/

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : convert packed byte to bit-array
* Prototype     : (as listed in header-file)
* Public/Local  : public  (public means prototype is in header-file)
* Parameters    : bits: int  (bit array)
* Return        : bytes: typebyte (byte array)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  dg
* Last Changes  :  19.04.97
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  19.04.97   dg     first
****************************************************************************/

void BC01_InputsToBit(void *InpBytes, void *InpBits, int n)
{
	BYTE *InpBytes_;
	BYTE *InpBits_;
	int i1,i2,i3;
	int Mask;

	if (n>MAX_INPBITS)
		n = MAX_INPBITS;

	InpBytes_ = InpBytes;
	InpBits_ = InpBits;
	i2 = 0;	// bit-number
	i3 = 0;	// byte-number
	Mask = 0x01;
	for (i1=0; i1<n; i1++)
	{
		InpBits_[i1] = ((InpBytes_[i3] & Mask) != 0);
		Mask <<= 1;
		i2++;
		if (i2>7 && i3<MAX_INPBYTES-1)
		{
			// next byte
			Mask = 0x01;
			i2 = 0;
			i3++;
		}
		//printf("E %d: %02x i3=%d\n",i1,Byte[i1],i3);
	}

}	// BC01_InputsToBit()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : convert bit-array to packed byte
* Prototype     : (as listed in header-file)
* Public/Local  : public  (public means prototype is in header-file)
* Parameters    : bytes: typebyte (byte array)
* Return        : bits: int (bit array)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  dg
* Last Changes  :  19.04.97
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  19.04.97   dg     first version
****************************************************************************/

void BC01_OutputsToByte(void *OutBits, void *OutBytes, int n)
{
	BYTE *OutBytes_;
	BYTE *OutBits_;
	int i1,i2,i3;
	int Mask;

	if (n>MAX_OUTBITS)
		n = MAX_OUTBITS;

	OutBytes_ = OutBytes;
	OutBits_ = OutBits;
	i2 = 0;	// bit-number
	i3 = 0;	// byte-number
	Mask = 0x01;
	OutBytes_[i3] = 0x00;
	for (i1=0; i1<n; i1++)
	{
		if (OutBits_[i1])
			OutBytes_[i3] |= Mask;
		Mask <<= 1;
		i2++;
		if (i2>7 && i3<MAX_OUTBYTES-1)
		{
			// next byte
			Mask = 0x01;
			i2 = 0;
			i3++;
			OutBytes_[i3] = 0x00;
		}
	}

}	// BC01_OutputsToByte()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : define and read retentive flags from flash memory
* Prototype     : (as listed in header-file)
* Public/Local  : public  (public means prototype is in header-file)
* Parameters    : size: int (number of flags bytes)
*               : m: M_type  (retentve flags)
* Return        : int (0=error, 1=ok)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        :  dg
* Last Changes  :  29.04.97
* Version       :  1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  29.04.97   dg     first version
*  1.01  07.05.97   dg     changed pointer type to void
*						   changed order of parameters
****************************************************************************/

int BC01_Flags_Init(void *M, int size)
{
	union REGS regs;
	struct SREGS sregs;
	//BC01_RentOp_Type RentOp;		// structure for rententive operators/flags

	if (size>1024)
		return 0;

	RentOp.Entries = 1;
	RentOp.Segment = FP_SEG(M);
	RentOp.x[0].Offs = FP_OFF(M);
	RentOp.x[0].Size = size;		// size in bytes
	RentOp.x[0].MaxSize = 1024;
	RentOp.x[0].ElemSize = 1;
	RentOp.x[0].Distance = 1;
	memset(M,0x01,sizeof(M_Type));

	// check if retentive operators are available
	regs.h.ah = 0xCF;				// function 0xCF
	regs.h.al = 0xFF;				// sub-function 0xFF
	int86(0x6F,&regs,&regs);
	if (regs.x.ax==0)
	{
		// available
		sregs.es  = FP_SEG(&RentOp);
		regs.x.bx = FP_OFF(&RentOp);
		regs.h.ah = 0xCF;			// function 0xCF
		regs.h.al = 0x00;			// sub-function 0x00
		int86x(0x6F,&regs,&regs,&sregs);
		return 1;
	}
	else
		// not available
		return 0;

}	// BC01_Flags_Init()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : write retentive flags to flash memory
*				  uses values from BC01_Flags_Read
* Prototype     : (as listed in header-file)
* Public/Local  : public  (public means prototype is in header-file)
* Parameters    : -/-
* Return        : int (0=error, 1=ok)
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg
* Last Changes  : 29.04.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  27.04.97   dg     first version
****************************************************************************/

int BC01_Flags_Write(void)
{
	union REGS regs;
	struct SREGS sregs;

	// check if retentive operators are available
	regs.h.ah = 0xCF;				// function 0xCF
	regs.h.al = 0xFF;				// sub-function 0xFF
	int86(0x6F,&regs,&regs);
	if (regs.x.ax==0)
	{
		// available
		sregs.es  = FP_SEG(&RentOp);
		regs.x.bx = FP_OFF(&RentOp);
		regs.h.ah = 0xCF;			// function 0xCF
		regs.h.al = 0xEE;			// sub-function 0xEE
		int86x(0x6F,&regs,&regs,&sregs);
		return 1;
	}
	else
		// not available
		return 0;

}	// BC01_Flags_Write()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : reset BC01_Cfg
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 09.06.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

void BC01_Image_CfgClr(void)
{
	memset(&BC01_Cfg,0x00,sizeof(BC01_Cfg_Type));
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Initialize process image
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

void BC01_Image_Init(void)
{
	int i;

	memset(&E,0x00,sizeof(E_Type));
	memset(&A,0x00,sizeof(A_Type));

	// remote
	if (BC01_Cfg.Remote>0)
	{
		i = BC01_Remote_Installed();
		switch(i)
		{
			case -2:
				BC01_Cfg.RemoteStat = 0;	// REO.EXE not installed (error)
				break;
			case -1:						// not initialized
				BC01_Cfg.RemoteStat = 1;
				//printf("init remote:\n");
				i = 0;
				while (i<2 && BC01_Cfg.RemoteStat==1)
				{
					if (BC01_Remote_Init(1))
						BC01_Cfg.RemoteStat = 4;		// master new
					else
						//printf("error init remote, try %d\n",i+1);
					i++;
				}
				break;
			case 0:
				BC01_Cfg.RemoteStat = 3;		// master old
				break;
			case 1:
				BC01_Cfg.RemoteStat = 2;		// slave (error)
				break;
		}
	} // switch()

	// init fast counters
	BC01_FastCounter_Enable();
	for (i=0; i<2; i++)
	{
		if (BC01_Cfg.Counter[i].Enable)
		{
			BC01_FastCounter_Init(i,BC01_Cfg.Counter[i].Mode,
								  BC01_Cfg.Counter[i].Max);
			BC01_FastCounter_Start(i);
		}
	}

	// init timer
	memset(Timer_Buffer,0x00,sizeof(Timer_Buffer));

	if (BC01_Timer_Installed())
	{
		BC01_Timer_Enable(Timer_Buffer,MAX_TIMER);
		// set timer 0 for cycle time control
		if (BC01_Cfg.CycleTime>0)
		{
			BC01_Cfg.Cycle.Enable = TRUE;
			BC01_Cfg.Cycle.Prescale = 1;
			BC01_Cfg.Cycle.Handle =
				BC01_Timer_Insert(0,0,BC01_Cfg.Cycle.Prescale);
			BC01_Timer_Start(BC01_Cfg.Cycle.Handle);
		}
		// set user timer 1..MAX_TIMER-1
		for (i=0; i<MAX_TIMER-1; i++)
			if (BC01_Cfg.Timer[i].Enable)
			{
				BC01_Cfg.Timer[i].Handle =
					BC01_Timer_Insert(0,0,BC01_Cfg.Timer[i].Prescale);
				BC01_Timer_Start(BC01_Cfg.Timer[i].Handle);
			}
	}

	// init potentiometer
	if (BC01_Cfg.TrimmerEnable)
		BC01_Trimmer_Enable();

	// set user-watchdog
	// if cycle time is longer than 480 ms, the user wazchdog will
	// not be activated
	if (BC01_Cfg.WatchdogEnable && BC01_Cfg.CycleTime<=100)
	{
		BC01_Watchdog_SetUser();
		BC01_Watchdog_Refresh();
	}

}	// BC01_Image_Init()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Deinitialize process image
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

void BC01_Image_Deinit(void)
{
	int i;

	if (BC01_Cfg.WatchdogEnable)
	{
		BC01_Watchdog_Refresh();
		BC01_Watchdog_SetBios();
	}

	// deinit fast counters
	BC01_FastCounter_Disable();
	for (i=0; i<2; i++)
	{
		if (BC01_Cfg.Counter[i].Enable)
			BC01_FastCounter_Stop(i);
	}

	// deinit trimmer
	if (BC01_Cfg.TrimmerEnable)
		BC01_Trimmer_Disable();

	// deinit timer
	BC01_Timer_Disable();

	// reset all outputs
	memset(&A,0x00,sizeof(A_Type));
	BC01_Image_Write();		// write process image

}	// BC01_Image_Deinit()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Read process image
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

void BC01_Image_Read(void)
{
	int i,r,mask;

	if (BC01_Cfg.WatchdogEnable)
		BC01_Watchdog_Refresh();

	// start cycle time measurement
	if (BC01_Cfg.Cycle.Enable)
		BC01_Timer_Write(BC01_Cfg.Cycle.Handle,0);

	memset(&E,0x00,sizeof(E_Type));
	//memset(&A.A_Bit,0x00,sizeof(BYTE)*MAX_OUTBITS);

	// read local digital inputs
	BC01_Inputs_ReadRemote((BYTE *) &(E.E_Byte[0]),2,0);
	//for (i=0; i<2; i++)
	//	BC01_BitToByte(E.E_Byte[i],E.E_Bytes[i]);

	// read local run/stop switch
	if (BC01_RunStop_ReadRemote(0))
		E.RunStop = TRUE;
	else
		E.RunStop = FALSE;

	// read fast counters
	for (i=0; i<2; i++)
	{
		if (BC01_Cfg.Counter[i].Enable)
			E.Count[i] = BC01_FastCounter_Read(i);
	}
	// read trimmer
	if (BC01_Cfg.TrimmerEnable)
		E.Trimmer = BC01_Trimmer_Read();

	// read timer
	for (i=0; i<MAX_TIMER; i++)
		if (BC01_Cfg.Timer[i].Enable)
			E.Timer[i] = BC01_Timer_Read(BC01_Cfg.Timer[i].Handle);

	// read remote digital inputs
	mask = 0x01;
	for (r=0; r<MAX_REMOTE; r++)
	{
		if (BC01_Cfg.Remote & mask)
		{
			// read remote digital inputs
			BC01_Inputs_ReadRemote((BYTE *) &(E.Remote[r].E_Byte),2,r+1);
			//for (i=0; i<2; i++)
			//	BC01_BitToByte(E.Remote[r].E_Byte[i],E.Remote[r].E_Bytes[i]);
			// read remote run/stop switch
			E.Remote[r].RunStop = (BC01_RunStop_ReadRemote(r+1) != 0);
		}
		mask <<= 1;
	}

	//if (BC01_Cfg.BitArrayEnable)
	//	BC01_Image_InputToBits();

}	// BC01_Image_Read()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Put Image bits into bit-array
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

void BC01_Image_InputToBits(void)
{
	int r,i;

	//printf("\nBC01_ImageToBits() Start\n");

	// local
	BC01_InputsToBit(E.E_Byte,E.E_Bit,MAX_INPBITS);

	// remote
	for (r=0; r<MAX_REMOTE; r++)
		for (i=0; i<MAX_INPBYTES; i++)
			BC01_InputsToBit(E.Remote[r].E_Byte,E.Remote[r].E_Bit,MAX_INPBITS);
	//printf("BC01_ImageToBits() End\n");
}

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Write process image
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

void BC01_Image_Write(void)
{
	int r,Mask;

	//if (BC01_Cfg.BitArrayEnable)
	//	BC01_Image_OutputToByte();

	if (BC01_Cfg.Cycle.Enable)
		while (BC01_Timer_Read(BC01_Cfg.Cycle.Handle)<BC01_Cfg.CycleTime);

	// write local data
	BC01_Outputs_WriteRemote((BYTE *) &(A.A_Byte),1,0);

	// set led
	BC01_Led_Write(A.Led & 0x07);

	Mask = 0x01;
	for (r=0; r<MAX_REMOTE; r++)
	{
		if (BC01_Cfg.Remote & Mask)
			BC01_Outputs_WriteRemote((BYTE *) &(A.Remote[r].A_Byte),1,r+1);
		Mask <<= 1;
	}

}	// BC01_Image_Write()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Put output bits form bitarray into byte
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

void BC01_Image_OutputToByte(void)
{
	int r;

	// local
	BC01_OutputsToByte(A.A_Bit,A.A_Byte,MAX_OUTBITS);

	// remote
	for (r=0; r<MAX_REMOTE; r++)
		BC01_OutputsToByte(A.Remote[r].A_Bit,A.Remote[r].A_Byte,MAX_OUTBITS);
}

//********************************************************************
//
//  BC01/BC02 Low level Communication Routines (Fossil-Driver)
//
//********************************************************************

#define FCTRL_XON_SEND 0x01
#define FCTRL_CTS 0x02
#define FCTRL_XON_RECEIVE 0x04

#define STAT_DCD 0x0080
#define STAT_RDA 0x0100
#define STAT_OVRN 0x0200
#define STAT_THRE 0x2000
#define STAT_TSRE 0x4000

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Check for installed com driver
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : -
* Return        : BOOL:
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

#ifdef FALSE
BOOL BC01_Com_Installed(void)
{
	BYTE far *p;
	WORD far *p1;

/*
  PtrSeg := MemW[0:$14*4+2];
  PtrOfs := MemW[0:$14*4];
  if MemW[PtrSeg:PtrOfs+6]=$1954 then
*/

	p = MK_FP(0x0000,0x14*4);
	p1 = (WORD far *)p;
	p = (BYTE far *) MK_FP(*(p1+1),*p1);
	p = p + 6;
	p1 = (WORD far *)p;
	if (*p1==0x1954)
	{
		BC01_Com_Result = 0;
		return TRUE;
	}
	else
	{
		BC01_Com_Result = 1;
		return FALSE;
	}

}	// BC01_Com_Installed()
#endif

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Set baud rate for com port
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: Port			Portnr
*				  int: Baud			Baudrate
*				  int: Parity		Parity
*				  int: Datenbits	Databits
*				  int: Stopbits		Stopbits
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

int BC01_Com_SetBaud(int Port,int Baud,int Parity,int Datenbits,int Stopbits)
{
	union REGS regs;
	const WORD BaudTab[8] = {19200u,38400u,300u,600u,1200u,2400u,4800u,9600u};
	int i;
	BYTE Param;

	if (Port!=BC01_COM_EXT)
		goto Fehler;

	Param = 0;
	for (i=0; i<8; i++)
	{
		if (Baud==BaudTab[i])
		{
			Param = i << 5;
			goto BaudOk;
		}
	}
	// ungueltige Baudrate
	goto Fehler;

BaudOk:
	if (Parity==2)
		Parity = 3;
	Param |= (Parity & 0x03) << 3;

	if (Stopbits<1 || Stopbits>2)
		goto Fehler;
	Param |= (Stopbits-1) << 2;

	if (Datenbits<5 || Datenbits>8)
		goto Fehler;
	Param |= (Datenbits-5) & 0x03;

	regs.h.ah = 0x00;  // function 0
	regs.x.dx = Port;
	regs.h.al = Param;
	int86(0x14,&regs,&regs);
	BC01_Com_Result = 0;
	return regs.x.ax;

Fehler:
	BC01_Com_Result = 5;
	return 0;

}	// BC01_Com_SetBaud()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Send Character and wait for send ready
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: port
*				  char: chr
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

int BC01_Com_SendWait(int Port, char Chr)
{
	union REGS regs;

	regs.h.ah = 0x01;	// Funktions-Nr
	regs.x.dx = Port;
	regs.h.al = Chr;
	int86(0x14,&regs,&regs);
	return regs.x.ax;

}	// BC01_Com_SendWait()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Get char from port with wait
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: port
* Return        : int:
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

int BC01_Com_GetWait(int Port)
{
	union REGS regs;

	regs.h.ah = 0x02;
	regs.x.dx = Port;
	int86(0x14,&regs,&regs);
	return regs.h.al;

}	// BC01_Com_GetWait()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : get status from com port
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: port
* Return        : WORD:
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

WORD BC01_Com_GetStatus(int Port)
{
	union REGS regs;

	regs.h.ah = 0x03;
	regs.x.dx = Port;
	int86(0x14,&regs,&regs);
	return regs.x.ax;

}	// BC01_Com_GetStatus()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Check for character available
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int:Port
* Return        : BOOL:
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

BOOL BC01_Com_ChrAvail(int Port)
{
	return (BC01_Com_GetStatus(Port) & STAT_RDA) != 0;

}	// BC01_Com_ChrAvail()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Initialize com port
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: port
* Return        : BOOL:
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

BOOL BC01_Com_Init(int Port)
{
	union REGS regs;

	regs.h.ah = 0x04;
	regs.x.dx = Port;
	regs.x.bx = 0;
	int86(0x14,&regs,&regs);
	if (regs.x.ax==0x1954)
	{
		//printf("Rev=%d\n",regs.h.bh);
		//printf("Max=%d\n",regs.h.bl);
		BC01_Com_Result = 0;
		return TRUE;
	}
	else
	{
		BC01_Com_Result = 2;
		return FALSE;
	}

}	// BC01_Com_Init()

//********************************************************************
//
//  BC01/BC0x Console I/O (Console/COM1/Extention)
//
//********************************************************************

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Get Char from Console
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: Channel
* Return        : int:
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

int BC01_GetChar(int Channel)
{
#ifdef __WATCOMC__
	union REGPACK regs;	
#else
	union REGS regs;
#endif

	switch(Channel)
	{
		case BC01_DEV_CON:
			regs.h.ah = 6;
			regs.h.dl = 0xFF;
#ifdef __WATCOMC__
			intr(0x21, &regs);
#else			
			int86(0x21, &regs, &regs);
#endif
			if (regs.x.flags & 0x40)
				return -1;
			else
				return regs.h.al;
		case BC01_DEV_EXT:
			if (BC01_Com_ChrAvail(BC01_COM_EXT))
				return BC01_Com_GetWait(BC01_COM_EXT);
			else
				return -1;
		case BC01_DEV_COM:
			if (BC01_Com_ChrAvail(BC01_COM_COM))
				return BC01_Com_GetWait(BC01_COM_COM);
			else
				return -1;
	}
	return 0;	// dummy

}	// BC01_GetChar()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Put char to console
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: Channel
*				  int: c
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

void BC01_PutChar(int Channel, int c)
{
	union REGS regs;

	if (c==0xFF)
		return;

	switch(Channel)
	{
		case BC01_DEV_CON:
			regs.h.ah = 6;
			regs.h.dl = c;
			int86(0x21, &regs, &regs);
			break;
		case BC01_DEV_EXT:
			BC01_Com_SendWait(BC01_COM_EXT, c);
			break;
		case BC01_DEV_COM:
			BC01_Com_SendWait(BC01_COM_COM, c);
	}

}	// BC01_PutChar()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Put string to  console
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: Channel
*				  char: *s string
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

void BC01_PutStr(int Channel, char *s)
{
	while (*s)
		BC01_PutChar(Channel,*s++);

}	// BC01_PutStr()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : set cursor to position x,y with ANSI codes
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: Channel
*				  BYTE: x
*				  BYTE: y
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/
void BC01_Ansi_CrsrXY(int Channel,BYTE x, BYTE y)
{
	char s[3];

	BC01_PutStr(Channel,"\x1B[");
	itoa(y+1,s,10);
	BC01_PutStr(Channel,s);
	BC01_PutChar(Channel,';');
	itoa(x+1,s,10);
	BC01_PutStr(Channel,s);
	BC01_PutChar(Channel,'H');

}	// BC01_Ansi_CrsrXY()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Set cursor in upper left corner of screen with ANSI-codes
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: Channel
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/
void BC01_Ansi_CrsrHome(int Channel)
{
	BC01_PutStr(Channel,"\x1B[1;1H");

}	// BC01_Ansi_CrsrHome()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Clear screen with ANSI-codes
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: Channel
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

void BC01_Ansi_ClrScr(int Channel)
{
	BC01_PutStr(Channel,"\x1B[2J");

}	// BC01_Ansi_ClrScr()

/****************************************************************************
*
* (C) 1997 by Beck Computer-Loesungen GmbH
*
* ---------------------------------------------------------------------------
* Function      : Clear to end of Line with ANSI-codes
* Prototype     : (as listed in header-file)
* Public/Local  : local  (public means prototype is in header-file)
* Parameters    : int: Channel
* Return        : -
* Header-File   : BC01.H
* ---------------------------------------------------------------------------
* Author        : dg/tw
* Last Changes  : 25.05.97
* Version       : 1.00
* ---------------------------------------------------------------------------
* History:
*
*  Vx.yy  Date     Author  Changes
*  1.00  25.05.97   dg     first version
****************************************************************************/

void BC01_Ansi_ClrEol(int Channel)
{
	BC01_PutStr(Channel,"\x1B[K");

}	// BC01_Ansi_ClrEol()

/****************************************************************************
* END OF BC01.C
****************************************************************************/
